package XCSphere;

/**
 * This class has been modified to use hypersphere classifiers instead of hyperplane classifiers.
 * This class handles the different sets of classifiers. It stores each set in an array. The array is initialized to 
 * a sufficient large size so that no changes in the size of the array will be necessary.
 * The class provides constructors for constructing 
 * <ul>
 * <li> the empty population,
 * <li> the match set, and
 * <li> the action set.
 * </ul>
 * It executes a GA in a set and updates classifier parameters of a set.
 * Moreover, it provides all necessary different sums and averages of parameters in the set.
 * Finally, it handles addition, deletion and subsumption of classifiers.
 *
 * @author    Martin V. Butz
 * @version   XCSJava 1.0
 * @since     JDK1.1
 */
public class XCSphereClassifierSet extends XClassifierSet
{
     /**
     * Creates a new, empty population initializing the population array to the maximal population size 
     * plus the number of possible actions.
     *
     * @see XCSConstants#maxPopSize
     * @param numberOfActions The number of actions possible in the problem.
     */
    public XCSphereClassifierSet(int numberOfActions)
    {
        numerositySum=0;
        cllSize=0;
        parentSet=null;
        clSet=new XClassifier[XCSphereConstants.maxPopSize+numberOfActions];
    }

    /**
     * Constructs a match set out of the population. After the creation, it is checked if the match set covers all possible actions
     * in the environment. If one or more actions are not present, covering occurs, generating the missing action(s). If maximal 
     * population size is reached when covering, deletion occurs.
     *
     * @see XClassifier#XClassifier(double,int,String,int)
     * @see XCSConstants#maxPopSize
     * @see #deleteFromPopulation
     * @param state The current situation/problem instance.
     * @paramn pop The current population of classifiers.
     * @param time  The actual number of instances the XCS learned from so far.
     * @param numberOfActions The number of actions possible in the environment.     
     */
    public XCSphereClassifierSet(String state, XClassifierSet pop, int time, int numberOfActions)
    {
        parentSet=pop;
        numerositySum=0;
        cllSize=0;
        clSet=new XClassifier[pop.cllSize+numberOfActions];

        boolean[] actionCovered =  new boolean[numberOfActions];
        for(int i=0; i<actionCovered.length; i++)
            actionCovered[i]=false;

        for(int i=0; i<pop.cllSize; i++){
            XClassifier cl=pop.clSet[i];
            if( cl.match(state)){
                addClassifier(cl);
                actionCovered[cl.getAction()]=true;
            }
        }

        //Check if each action is covered. If not -> generate covering XClassifier and delete if the population is too big
        boolean again;
        XClassifier newCl;
        do{
            again=false;
            for(int i=0; i<actionCovered.length; i++){
                if(!actionCovered[i]){
                	if (XCSphereConstants.nextDoubleFromUniformDist(0, 1) < XCSphereConstants.spherePlaneSplit)
                	{
                		newCl=new XClassifier(numerositySum+1, time, state, i);
                	}
                	else
                	{
                		newCl=new XCSphereClassifier(numerositySum+1, time, state, i);
                	}
                    
                    addClassifier(newCl);
                    pop.addClassifier(newCl);
                }
            }
            while(pop.numerositySum > XCSphereConstants.maxPopSize){
                XClassifier cdel=pop.deleteFromPopulation();
                // update the current match set in case a classifier was deleted out of that 
                // and redo the loop if now another action is not covered in the match set anymore.
		int pos=0;
                if(cdel!=null && (pos=containsClassifier(cdel))!=-1) {
		    numerositySum--;
		    if(cdel.getNumerosity()==0){
			removeClassifier(pos);
			if( !isActionCovered(cdel.getAction())){
			    again=true;
			    actionCovered[cdel.getAction()]=false;
			}
		    }
		}
            }
        }while(again);
    }


    /**
     * Constructs an action set out of the given match set. 
     * 
     * @param matchSet The current match set
     * @param action The chosen action for the action set.
     */
    public XCSphereClassifierSet(XClassifierSet matchSet, int action)
    {
        parentSet=matchSet;
        numerositySum=0;
        cllSize=0;
        clSet=new XClassifier[matchSet.cllSize];
     
        for(int i=0; i<matchSet.cllSize; i++){
            if( matchSet.clSet[i].getAction() == action){
                addClassifier(matchSet.clSet[i]);
            }
        }
    }

    /**
     * The Genetic Discovery in XCS takes place here. If a GA takes place, two classifiers are selected
     * by roulette wheel selection, possibly crossed and mutated and then inserted.
     *
     * @see XCSConstants#theta_GA
     * @see #selectXClassifierRW
     * @see XClassifier#twoPointCrossover
     * @see XClassifier#applyMutation
     * @see XCSConstants#predictionErrorReduction
     * @see XCSConstants#fitnessReduction
     * @see #insertDiscoveredXClassifiers
     * @param time  The actual number of instances the XCS learned from so far.
     * @param state  The current situation/problem instance.
     * @param numberOfActions The number of actions possible in the environment.
     */ 
    public void runGA(int time, String state, int numberOfActions)
    {
        // Don't do a GA if the theta_GA threshold is not reached, yet
        if( cllSize==0 || time-getTimeStampAverage() < XCSphereConstants.theta_GA )
            return;
    
        setTimeStamps(time);

        double fitSum=getFitnessSum(); 
        // Select two XClassifiers with roulette Wheel Selection
        XClassifier cl1P=selectXClassifierRW(fitSum);
        XClassifier cl2P=selectXClassifierRW(fitSum);
    
        XClassifier cl1=(XClassifier) cl1P.clone();
        XClassifier cl2=(XClassifier) cl2P.clone();

    	// only apply crossover if classifiers are of same class, otherwise just clone and mutate
        if (cl1.getClass().equals(cl2.getClass()))
        {
        	cl1.twoPointCrossover(cl2);
        }

        cl1.applyMutation(state, numberOfActions);
        cl2.applyMutation(state, numberOfActions);
        
        cl1.setPrediction((cl1.getPrediction() + cl2.getPrediction())/2.);
        cl1.setPredictionError(XCSphereConstants.predictionErrorReduction * (cl1.getPredictionError() + cl2.getPredictionError())/2.);
        cl1.setFitness(XCSphereConstants.fitnessReduction * (cl1.getFitness() + cl2.getFitness())/2.);
        cl2.setPrediction(cl1.getPrediction());
        cl2.setPredictionError(cl1.getPredictionError());
        cl2.setFitness(cl1.getFitness());
   
        insertDiscoveredXClassifiers(cl1, cl2, cl1P, cl2P);
    }
}
