package XCSphere;

import java.io.PrintWriter;

/**
 * A hypersphere classifier to replace the standard LCS hyperplane classifier
 *  
 * @author marshall
 *
 */
public class XCSphereClassifier extends XClassifier
{
	/** The radius of the hypersphere classifier */
	private int radius;
	
	// TODO: should radii have possibility to be exchanged during crossover?
	
	/**
     * Creates a condition randomly considering the constant <code>radius<\code>.
     *
     * @see XCSConstants#radius
     * @see XCSphereConstants#newRadiusMean
     * @see XCSphereConstants#newRadiusVariance
     * 
     */
	protected void createRandomCondition(int condLength)
	{
		char condArray[]=new char[condLength];
		for(int i=0; i < condLength; i++){
		    if(XCSphereConstants.drand() < 0.5)
		    {
			    condArray[i]='0';
		    }
			else
			{
			    condArray[i]='1';
			}
		}
		condition = new String(condArray);		
	}
	
	/**
     * Mutates the condition of the classifier. If one allele is mutated depends on the constant pM. 
     * This mutation is a niche mutation. It assures that the resulting classifier
     * still matches the current situation.
     *
     * @see XCSConstants#pM
     * @see XCSphereConstants#radiusMutationProb
     * @see XCSphereConstants#radiusMutationVariance
     * 
     * @param state The current situation/problem instance.
     * 
     */
    protected boolean mutateCondition(String state)
    {
		boolean changed = false;
		int newRadius, condLength = condition.length();
	    
		for(int i=0; i<condLength; i++)
		{
		    if(XCSphereConstants.drand() < XCSphereConstants.pM)
		    {
				char[] cond = condition.toCharArray();
				changed = true;
				if(cond[i] == '0')
				{
				    cond[i] = '1';
				}
				else
				{
				    cond[i] = '0';
				}
				condition = new String(cond);
		    }
		}
		if (XCSphereConstants.drand() < XCSphereConstants.radiusMutationProb)
		{
			newRadius = radius + XCSphereConstants.nextIntFromNormalDist(0, XCSphereConstants.radiusMutationStdDev);
			if (newRadius < 0)
			{
				newRadius = 0;
			}
			if (newRadius > condition.length())
			{
				newRadius = condition.length();
			}
			if (newRadius != radius)
			{
				radius = newRadius;
				changed = true;
			}
		}
		
		return changed;
    }

    /**
     * Creates a matching condition
     *
     * @param cond
     * 
     */
    protected void createMatchingCondition(String cond)
    {
		condition=new String(cond);
    }
    
    /**
     * Returns if the classifier matches in the current situation.
     *
     * @param state The current situation which can be the current state or problem instance.
     * 
     */
    public boolean match(String state)
    {
    	return (distance(state) <= radius);
    }

	/**
	 * Returns the strength of the classifier's match in the current situation.
	 *
	 * @param state The current situation which can be the current state or problem instance.
	 * 
	 */
	public int matchStrength(String state)
	{
		return 1;
	}


	/**
	 * Returns the Hamming distance from the classifier's centre to the current situation.
	 *
	 * @param state The current situation which can be the current state or problem instance.
	 * 
	 */
	public int distance(String state)
	{
		int l1, hammingDistance = 0;
    	
/*    	System.out.println(state);
		System.out.println(condition);
		System.out.print(radius + ": ");
		if(condition.length()!=state.length())
		{
			System.out.println("no match");
			return false;
		}*/
		for (l1 = 0; l1 < condition.length(); l1++)
		{
			if (condition.charAt(l1) != state.charAt(l1))
			{
				hammingDistance++;
			}
		}
/*    	if (numDiffs <= radius)
		{
			System.out.println("match");
		}
		else
		{
			System.out.println("no match");
		}*/
    	
		return (hammingDistance);
	}

    /**
     * Returns if the classifier is more general than cl. It is made sure that the classifier is indeed more general and 
     * not equally general as well as that the more specific classifier is completely included in the more general one 
     * (do not specify overlapping regions). For hyperspheres with different centres this can be tested by confirming that
     * the distance between the centres of the two hyperspheres plus the radius of the less general hypersphere is less
     * than the radius of the more general hypersphere (i.e. d_lm + r_l <= r_m)
     *
     * @param The classifier that is tested to be more specific.
     * 
     */
    public boolean isMoreGeneral(XClassifier cl)
    {
    	int l1, matchRadius, hammingDistance = 0;
    	XCSphereClassifier sphere;

    	if (!cl.getClass().equals(this.getClass()))
		{
    		return false;
		}
    	sphere = (XCSphereClassifier) cl;
    	if (condition.equals(cl.condition))
    	{
    		return (sphere.radius < radius);
    	}
    	else
    	{
	    	for (l1 = 0; l1 < cl.condition.length(); l1++)
	    	{
	    		if (condition.charAt(l1) != cl.condition.charAt(l1))
	    		{
	    			hammingDistance++;
	    		}
	    	}
	    	matchRadius = Math.min(hammingDistance + sphere.radius, condition.length());
	    	
	    	if (radius == sphere.radius)
	    	{
	    		// if hypersphere radii are the same then one cannot be more general than the other
	    		return false;
	    	}
	    	else
	    	{
	    		return (matchRadius <= radius);
	    	}
    	}
    }
    
    /**
     * Returns if the two classifiers are identical in condition and action.
     *
     * @param cl The classifier to be compared.
     * 
     */
    public boolean equals(XClassifier cl)
    {
    	if (!cl.getClass().equals(this.getClass()))
		{
    		return false;
		}
    	XCSphereClassifier sphere = (XCSphereClassifier) cl;
    	
    	return (radius == sphere.radius && super.equals(cl));
    }
    
    /**
     * Constructs a classifier with matching condition and specified action.
     *
     * @param setSize The size of the current set which the new classifier matches.
     * @param time  The actual number of instances the XCS learned from so far.
     * @param situation The current problem instance/perception.
     * @param act The action of the new classifier.
     * 
     */
    public XCSphereClassifier(double setSize, int time, String situation, int act)
    {			
    	super(setSize, time, situation, act);
    	radius = XCSphereConstants.nextIntFromNormalDist(XCSphereConstants.newRadiusMean, XCSphereConstants.newRadiusStdDev);
    	if (radius < 0)
    	{
    		radius = 0;
    	}
    }
  
    /**
     * Construct matching classifier with random action.
     *
     * @param setSize The size of the current set which the new classifier matches.
     * @param time The actual number of instances the XCS learned from so far.
     * @param numberOfActions The number of different actions to chose from 
     * (This should be set to the number of actions possible in the problem).
     * @param situation The current problem instance/perception.
     */
    public XCSphereClassifier(double setSize, int time, int numberOfActions, String situation)
    {
    	super(setSize, time, numberOfActions, situation);
    	radius = XCSphereConstants.nextIntFromNormalDist(XCSphereConstants.newRadiusMean, XCSphereConstants.newRadiusStdDev);
    	if (radius < 0)
    	{
    		radius = 0;
    	}
    }
	
    /**
     * Construct a classifier with random condition and random action.
     *
     * @param setSize The size of the current set which the new classifier matches.
     * @param time  The actual number of instances the XCS learned from so far.
     * @param condLength The length of the condition of the new classifier.
     * @param numberOfActions The number of different actions to chose from 
     */ 
    public XCSphereClassifier(double setSize, int  time, int condLength, int numberOfActions)
    {
    	super(setSize, time, condLength, numberOfActions);
    	radius = XCSphereConstants.nextIntFromNormalDist(XCSphereConstants.newRadiusMean, XCSphereConstants.newRadiusStdDev);
    	if (radius < 0)
    	{
    		radius = 0;
    	}
    }
  
    /**
     * Constructs an identical XCSphereClassifier.
     * However, the experience of the copy is set to 0 and the numerosity is set to 1 since this is indeed 
     * a new individual in a population.
     *
     * @param clOld The to be copied classifier.
     */
    public XCSphereClassifier(XClassifier clOld)
    {
    	super(clOld);
    	radius = ((XCSphereClassifier) clOld).radius;
    }

    /**
     * Clones the XCSphereClassifier
     * 
     * @return cloned XCSphereClassifier
     * 
     */
    protected Object clone()
    {
    	return new XCSphereClassifier(this);
    }

    
    /**
     * Prints the classifier to the control panel.
     * The method prints condition radius action prediction predictionError fitness numerosity experience actionSetSize timeStamp.
     */
    public void printXClassifier()
    {
    	System.out.println(condition+" "+radius+" "+action+"\t"+(float)prediction+"\t"+(float)predictionError+"\t"+(float)fitness+
			   "\t"+numerosity+"\t"+experience+"\t"+(float)actionSetSize+"\t"+timeStamp);
    }
  
    /**
     * Prints the classifier to the print writer (normally referencing a file).
     * The method prints condition radius action prediction predictionError fitness numerosity experience actionSetSize timeStamp.
     *
     * @param pW The writer to which the classifier is written.
     */
    public void printXClassifier(PrintWriter pW)
    {
    	pW.println(condition+"-"+radius+"-"+action+" "+(float)prediction+" "+(float)predictionError+" " +(float)fitness+
		   " "+numerosity+" "+experience+" "+(float)actionSetSize+" "+timeStamp);		
    }	
}