/* Abstract superclass for Boolean functions. 
 * 
 * It provides randomly generated problem instances and 
 * interfaces with the rest of the system.
 * Any length of function can be used.
 * 
 * Subclasses must only implement isCorrect() which implements a particular Boolean
 * function (or set of them) by returning true or false on the current instance.
 *
 * Based on Martin Butz's MPEnvironment.java.
 * Note that unlike MPEnvironment this does not support layered payoff landscapes.
 * 
 * @author    Tim Kovacs
 */

package XCSphere;

public abstract class BooleanEnvironment implements Environment {
    /**
     * Specifies the length of each presented problem. 
     */
    public static int conLength;
    
    /**
     * Specifies the maximal payoff possible in this environment.
     */
    protected int maxPayoff = 1000;    
    
    /**
     * Stores the current problem state.
     */
    protected char[] currentState;

    
    /*#####---- The following parameters are used internally ----#####*/
    
    /**
     * Stores if the last classification was correct.
     */
    protected boolean correct;
    
    /**
     * Is set to true after a classification was executed
     */
    protected boolean reset;

    /**
     * In Boolean problems there are two classifications possible!
     */
    protected final int nrActions=2;

    /**
     * Stores which states have been seen.
     */
    protected boolean[] statesSeen;

    /**
     * Constructs the Boolean environment according to the specified problem length. 
     * The relevant constants for the environment are calculated here 
     * and the array object storing the current state is created
     *
     * @param length Specifies the problem length.
     * @param trackInstancesSeen Enables counting of unique problem instances seen (only suitable for small problems) 
     */
    public BooleanEnvironment(int length, boolean trackInstancesSeen)
    {
	conLength=length;
	currentState = new char[conLength];
	correct=false;
	reset=false;
	statesSeen = null;
	if (trackInstancesSeen)
	{
		//create array to keep track of number of instances seen so far
		int arrayLength = (int) Math.pow(2,length);
		statesSeen = new boolean[arrayLength];
		for (int l1 = 0; l1 < arrayLength; l1++)
		{
			statesSeen[l1] = false;
		}
	}
    }

    /**
     * Returns number of unique instances seen so far
     *  
     * @return number of unique instances seen so far
     * 
     */
    public int numInstancesSeen()
    {
    	if (statesSeen != null)
    	{
	    	int numInstancesSeen = 0;
	    	int arrayLength = (int) Math.pow(2,conLength);
	    	for (int l1 = 0; l1 < arrayLength; l1++)
	    	{
	    		if (statesSeen[l1] == true)
	    		{
	    			numInstancesSeen++;
	    		}
	    	}
	    	
	    	return numInstancesSeen;
    	}
    	else
    	{
    		return 0;
    	}
    }

    /**
     * Returns proportion of instance space seen so far
     * 
     * @return proportion of instance space seen so far (in [0,1])
     * 
     */
    public double propInstancesSeen()
    {
    	return (double) numInstancesSeen() / Math.pow(2,conLength);
    }


    /**
     * Generates a new random problem instance.
     */
    public String resetState()
    {
	for(int i=0; i<conLength; i++){
	    if(XCSphereConstants.drand()<0.5){
		currentState[i]='0';
	    }else{
		currentState[i]='1';
	    }
	}
	reset=false;
	String newState = new String(currentState);
	if (statesSeen != null)
	{
		statesSeen[Integer.parseInt(newState,2)] = true;
	}	
	
	return newState;
    }
  
    /**
     * Calls isCorrect() and determines the reward.
     *
     * @param action Specifies the classification.
     */
    public double executeAction(int action)
    {
    	reset = true; // a signal to other classes
    	
    	if(isCorrect(action) == true){
    		correct=true;
    		return maxPayoff;
    	}else{
    	    correct=false;
    		return 0.0;
    	}
    }

    /**
     * Implemented by subclasses to define Boolean function(s)
     * @param action
     * @return true or false depending on whether currentState is correctly
     * classified by action or not
     */
    public abstract boolean isCorrect(int action);
    

    /**
     * Returns true if the last executed action was a correct classification
     */
    public boolean wasCorrect()
    {
	return correct;
    }

    /**
     * Returns false since the Multiplexer problem is a single step problem
     */
    public boolean isMultiStepProblem()
    {
	return false;
    }

    /**
     * Returns true after the current problem was classified
     */
    public boolean doReset()
    {
	return reset;
    }

    /**
     * Returns the problem length
     */
    public int getConditionLength()
    {
	return conLength;
    }

    /**
     * Returns the maximal payoff possible in the current multiplexer problem.
     */
    public int getMaxPayoff()
    {
	return maxPayoff;
    }

    /**
     * Returns the number of possible actions
     */
    public int getNrActions()
    {
	return nrActions;
    }

    /**
     * Returns the current state
     */
    public String getCurrentState()
    {
	return new String(currentState);
    }
}
