%    This file is part of UCS.
%
%    UCS is free software: you can redistribute it and/or modify
%    it under the terms of the GNU General Public License as published by
%    the Free Software Foundation, either version 3 of the License, or
%    (at your option) any later version.
%
%    UCS is distributed in the hope that it will be useful,
%    but WITHOUT ANY WARRANTY; without even the implied warranty of
%    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%    GNU General Public License for more details.
%
%    You should have received a copy of the GNU General Public License
%    along with UCS.  If not, see <http://www.gnu.org/licenses/>.


% Train ucs online given input,class tuples
% [pop trErr teErr] = ucstrain(pop,x,cl,maxIter,teX,teC)
% I/P arguments
%   pop - population of rules
%   x - input bit strings represented as an array of NxD where D is the
%       length of a string and N is the number of training points
%   cl - 0/1 class labels
%   maxIter - number of iterations over the entire training data
%   teX - input strings for the test dataset
%   teC - class labels of the test dataset
% O/P arguments
%   pop - modified population after the training
%   trErr - training error after each epoch
%   teErr - test error after each epoch
function [pop trErr teErr] = ucstrain(pop,x,cl,maxIter,teX,teC)
[N D] = size(x);

teErr = [];

startIter = pop.iter;

% start iterating over the data
for i = (startIter+1):(startIter + maxIter)
    currentIteration = mod(i,N) + 1;
    
    % find the matchset
    matchset = match(pop,x(currentIteration,:));
    pop.numMatches(matchset) = pop.numMatches(matchset) + 1;
    
    % find the correctset
    correctset = intersect(matchset,find(pop.action == cl(currentIteration)));
    avgLastTimeinGA = 0;
    
    if (isempty(correctset))
        pop = covering(pop,x(currentIteration,:),cl(currentIteration));
    else
        setSize = pop.correctSetSize(correctset);
        numCorrect = pop.numCorrect(correctset);
        %pop.correctSetSize(correctset) = (setSize.*numCorrect + length(correctset))./(numCorrect + 1);
        
        discount = (pop.beta).*ones(1,length(pop.n));
        indLess = find(pop.numMatches < 1/(pop.beta));
        discount(indLess) = 1./(pop.numMatches(indLess));
        pop.correctSetSize(correctset) = setSize + ...
            discount(correctset).*(sum(pop.n(correctset)) - setSize);
        
        pop.numCorrect(correctset) = numCorrect + 1;
        if (size(pop.lastTimeinGA,2) < max(correctset))
            disp('Something wrong');
        end;
        
        avgLastTimeinGA = sum(pop.lastTimeinGA(correctset).*pop.n(correctset))/sum(pop.n(correctset));
    end;
    
    % GA
    gaRecency = i - avgLastTimeinGA;
    if (gaRecency > pop.gaThreshold && ...
            length(correctset) > 0)
        %disp(sprintf('%d : invoking GA',i));
        pop.lastTimeinGA(correctset) = i;
        pop = invokeGA(pop, correctset);
    end;
    
    cpred = ucspredict(pop,x);
    trErr(i) = 1 - (sum(cpred == cl)/length(cl));
    if (exist('teX','var'))
        cpredT = ucspredict(pop,teX);
        teErr(i) = 1 - (sum(cpredT == teC)/length(teC));
    end;

    pop.iter = i;
end;
