package ucs;

import java.io.*;
import java.util.StringTokenizer;
import java.util.ArrayList;

/** Configuration object for UCS. You must make one of these before you can run UCS.
 * 
 * @author Gavin Brown
 */
public class UCSconfig implements Cloneable
{
	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////

	/** Random number seed. */
	protected static java.util.Random generator = new java.util.Random();
	
	/** Maximum population size to allow. This is in terms of <i>microclassifiers</i> as this implementation of UCS does not
	 * currently support macroclassifiers.  Default is <b>POPMAXSIZE = 400</b>. */
	public int POPMAXSIZE=400;	

	/** Online learning or offline learing. If this flag is set <b>true</b> the train and test sets are the same and testing will occur
	 * by periodically evaluating the system over the last 50 examples it has seen. Default is <b>onlinelearning = TRUE</b>. */
	public boolean onlinelearning=true;
		
	/** Probability of class noise being added to each example in the training data.
	* (only applies when doing <i>offline</i> learning).  Default is <b>noise = 0.0</b>.  */
	public double noise=0.0;
	
	/** Parameter controlling fitness evaluation for UCS, <i>acc<sup>v</sup></i>.  Default is <i>v = 20</i>*/
	public int v=20;

//PARAMS FOR THE GA COMPONENT

	/** Parameter controlling fitness for the GA in UCS, <i>acc<sup>v</sup></i>.  Default is <i>GA_v = v</i>*/
	public int GA_v=v;    

	/** The GA threshold.  Default is <i>gaThreshold = 25</i>. */
	public int gaThreshold = 25;
	
	/** The probability of mutating a single point in a rule condition.   Default is <i>mutationProb = 0.05</i>.  */
	public double mutationProb = 0.05;
	
	/** The probability of choosing crossover instead of mutation to perform on a rule condition. Default is <i>crossoverProb = 0.8</i>. */
	public double crossoverProb = 0.8;

	/** The experience threshold under which the accuracy is discounted.  Default is <i>inexperienceThreshold = 10</i>. */
	public int inexperienceThreshold = 10;
	
	/** The factor by which to discount when experience is too low. Default is <i>inexperiencePenalty = 0.01</i>. */
	public double inexperiencePenalty = 0.01;
	
	/** Probabililty of covering.   Default is <i>coveringProbability = 0.33333333</i>. */
	public double coveringProbability = 0.33333333;
	
	/** Subsumption experience threshold. Default is <i>ThetaSub = 20</i>. */
	public int ThetaSub = 20;
	public double ThetaSubAccuracyMinimum = 0.99;
	
	/** Deletion vote experience threshold.  Default is <i>ThetaDel = 20</i>. */
	public int ThetaDel = 20;

	/** Deletion vote fraction.  Default is <i>ThetaDelFrac = 0.10</i>. */
	public double ThetaDelFrac = 0.10;
		
	/** Object specifying the fitness function. By default this is a <b>FitnessFunction</b> object. */
	public FitnessFunction fitfunc = null;

	/** Object specifying the fitness function for the GA. By default this is a <b>FitnessFunction</b> object.
	    In standard UCS the same fitness function is used for both the GA and prediction. */
	public FitnessFunction GAfitfunc = null;
	
	/** Object specifying the system predictor. By default this is a <b>SystemPredictor</b> object. */
	public SystemPredictor systemPredictor = null;
	
	/** String specifying the file to source <b>training</b> data from. */
	public String trainingFile=null;
	
	/** String specifying the file to source <b>testing</b> data from. */
	public String testingFile=null;

        /** Number of folds to be used in cross-validation.*/
	public int folds=1;

        /** The current fold to use for testing. Initially set to -1, which signals that the inital folding has not been performed yet. */
	public int currentFold= -1;

        /** An ArrayList of ArrayLists. Each sub-arraylist contains a fold of the data */
	public ArrayList<ArrayList> allFolds;        

        /** Controls level of text output. 0 = minimal 1 = results for each test phase 2 = everything */
        public int verbosity=2;

	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////
	
	/** Sets the fitness function for the GA and prediction to the same supplied object.
	 * @param ff A fitness function object, either type <b>FitnessFunction</b> or an extension of it.
	 */
	public void setFitnessFunction ( FitnessFunction ff )
	{
		fitfunc = ff;
		GAfitfunc = ff;
	}

	/** Sets the fitness function for the GA and for prediction to the respective supplied objects.
	 * @param ff A fitness function object, either type <b>FitnessFunction</b> or an extension of it.
	 */
        public void setFitnessFunction ( FitnessFunction ff_pred, FitnessFunction ff_GA )
	{
		fitfunc = ff_pred;
		GAfitfunc = ff_GA;
	}

	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////

	/** Sets the system predictor to the supplied object.
	 * @param sp A system predictor object, either type <b>SystemPredictor</b> or an extension of it.
	 */
	public void setSystemPredictor ( SystemPredictor sp )
	{
		systemPredictor = sp;
	}

	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////
	
	/** Sets the training datafile name to the supplied argument. Note that since no testing datafile is specified,
	 * the system makes some assumptions for you.
	 * If <b>onlinelearning=TRUE</b>, then testing error will be reported over the last 50 examples seen.
	 * If <b>onlinelearning=FALSE</b>, then testing error will be obtained by a 50/50 random split of the training data.
	 * @param train The filename from which to load data.
	 */
	public void setProblem ( String train )
	{
		setProblem(train, null);
	}
	
	/** Sets the datafile names to the supplied arguments.
	 * @param train The filename from which to load training data.
	 * @param test The filename from which to load testing data.
	 * If <b>onlinelearning=TRUE</b>, the testing datafile is ignored.
	 */
	public void setProblem ( String train, String test )
	{
		trainingFile = train+"";
		if (test!=null) testingFile = test+"";
	}
	
	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////

	public UCSconfig()
	{
		fitfunc = new FitnessFunction();
		GAfitfunc = fitfunc;
		systemPredictor = new SystemPredictor();
	}
	
	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////
	
	/**
	 * Loads a UCSparameters configuration from the specified file.
	 * @param filename The file from which to load the configuration.
	 */
	public void load(String filename)
	{
		BufferedReader myReader = null;
		try {
			myReader = new BufferedReader( new FileReader( new File( filename ) ) );
		}
		catch( FileNotFoundException ex ) {
			System.err.println( "UCS params file '"+filename+"' not found." );
			System.exit(1);
		}
			
		try	{
		String line=null;
		StringTokenizer myTokenizer = null;
		while (myReader.ready())
		{	
				line = myReader.readLine();
				myTokenizer = new StringTokenizer( line, ", \t" );
				while(myTokenizer.countTokens() < 2) {
					line = myReader.readLine();
					myTokenizer = new StringTokenizer( line, ", \t" );
				}
				
				String p = (myTokenizer.nextToken()).toLowerCase();
				String value = myTokenizer.nextToken();
				
			if(p.equals("popmaxsize"))					POPMAXSIZE = Integer.parseInt(value);
			else if(p.equals("noise"))					noise = Double.parseDouble(value);
			else if(p.equals("v"))						v = Integer.parseInt(value);
			else if(p.equals("gathreshold"))			gaThreshold = Integer.parseInt(value);
			else if(p.equals("inexperiencethreshold"))	inexperienceThreshold = Integer.parseInt(value);
			else if(p.equals("inexperiencepenalty"))	inexperiencePenalty = Double.parseDouble(value);
			else if(p.equals("thetasub"))				ThetaSub = Integer.parseInt(value);
			else if(p.equals("thetadel"))				ThetaDel = Integer.parseInt(value);
			else if(p.equals("thetadelfrac"))			ThetaDelFrac = Double.parseDouble(value);
			else if(p.equals("thetasubaccuracyminimum"))ThetaSubAccuracyMinimum = Double.parseDouble(value);
			else if(p.equals("mutationprob"))			mutationProb = Double.parseDouble(value);
			else if(p.equals("crossoverprob"))			crossoverProb = Double.parseDouble(value);
			else if(p.equals("coveringprobability"))	coveringProbability = Double.parseDouble(value);
			else if (p.equals("onlinelearning")) {
				if(value.equals("true"))				onlinelearning=true;
				if(value.equals("false"))				onlinelearning=false;
			}
		}
		} catch (IOException ioe) {
			System.err.println( "IO Exception when reading datafile '"+filename+"'." );
			System.exit(1);
		}
		
	}

	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////

	/**
	 * Prints out configuration of the current UCSparameters instance.
	 */
	public void print(String prefix)
	{
			System.out.println(prefix + "POPMAXSIZE\t\t"+POPMAXSIZE);
			System.out.println(prefix + "onlinelearning\t\t"+onlinelearning);
			System.out.println(prefix + "noise\t\t\t"+noise);
			System.out.println(prefix + "v\t\t\t"+v);
			System.out.println(prefix + "GA_v\t\t\t"+GA_v);
			System.out.println(prefix + "gaThreshold\t\t"+gaThreshold);
			System.out.println(prefix + "mutationProb\t\t"+mutationProb);
			System.out.println(prefix + "crossoverProb\t\t"+crossoverProb);
			System.out.println(prefix + "inexperienceThreshold\t"+inexperienceThreshold);
			System.out.println(prefix + "inexperiencePenalty\t"+inexperiencePenalty);
			System.out.println(prefix + "coveringProbability\t"+coveringProbability);
			System.out.println(prefix + "ThetaSub\t\t"+ThetaSub);
			System.out.println(prefix + "ThetaSubAccuracyMinimum\t"+ThetaSubAccuracyMinimum);
			System.out.println(prefix + "ThetaDel\t\t"+ThetaDel);
			System.out.println(prefix + "ThetaDelFrac\t\t"+ThetaDelFrac);
			System.out.println();
	}
	
	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////
	
	/**
	 * Clones this UCSConfig object. It returns a shallow copy: primitive fields are copied
         * but objects fields are shared between the original and the clone. 
	 */
	public Object clone()
	{
	    try {
		return super.clone();
	    }
	    catch(CloneNotSupportedException e) {
		return null;
	    }
	}

	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////

}
