package ucs;

import dataprocessing.Example;

/**
 * A rule object for use in UCS.
 * @author Gavin Brown
 */
public class RuleCondition implements Cloneable
{	
	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////

	/** The condition */
	public double [] values;

	/** The arity of each element of the current condition. */
	public static int arity[];
	
	/** The length of this condition. */
	public int length;
	
	/** An integer constant, used to indicate the "don't care" symbol. */
	final static double HASH = -1;
	
	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////

	/**
	 * RuleCondition constructor.
	 * @param length The required length of this condition.
	 */
	public RuleCondition( int length )
	{
		this.length = length;
		arity = new int[length];
		
		// -1=#, 0=0, 1=1
		values = new double[length];
		for (int i=0; i<values.length; i++)
		{
			arity[i] = dataprocessing.Example.inputAlphabet[i].size();

			// generate random number from the legal range for this index
			//
			values[i] = randomValue(i);
			//values[i] = HASH;
		}
	}
	
	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////

	/**
	 * RuleCondition constructor.
	 * @param cond A double array specifying the contents of this condition.
	 */
	public RuleCondition( double [] cond )
	{
		this.length = cond.length;
		this.values = (double[])cond.clone();
	}

	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////
	
	/**
	 * Checks whether two conditions match or not.
	 * @param other A condition to check against this condition.
	 * @return True if the two conditons match, otherwise false.
	 */
	public boolean matches( RuleCondition other )
	{
		if(other.length!=this.length) return false;

		for (int i=0; i<values.length; i++)
		{
			if (values[i]!=HASH && values[i]!=other.values[i])
			{
				return false;
			}
		}

		
		return true;
	}
	
	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////
	
	/**
	 * Checks whether two conditions match or not.
	 * @param othergenome A double array specifying a condition to check against this condition.
	 * @return True if the two conditons match, otherwise false.
	 */
	public boolean matches( double [] othergenome )
	{
		if(othergenome.length!=values.length) return false;

		for (int i=0; i<values.length; i++)
		{
			if (values[i]!=HASH && values[i]!=othergenome[i])
			{
				return false;
			}
		}

		return true;
	}
	
	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////

	/**
	 * Sets this condition to the supplied argument.
	 * @param newgenome
	 */
	public void set( double [] newgenome )
	{
		values = (double[])newgenome.clone();
	}
	
	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////

	/**
	 * Sets a particular position in this condition to the supplied value.
	 * @param position The index within this condition to set.
	 * @param value A double value specifying what value the position should be set to. This could be Condition.HASH to set it as a "don't care" symbol.
	 */
	public void set( int position, double value )
	{
		values[position] = value;
	}
	
	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////

	/**
	 * Returns a valid randomly chosen value for the specified index in this rule, including the ``don't care'' symbol.
	 * @param position The index within this condition to query.
	 */
	public double randomValue( int position )
	{
		return UCSconfig.generator.nextInt( RuleCondition.arity[position]+1 )-1;
	}

	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////

	/**
	 * Returns a clone of this object.
	 * @return A clone of this RuleCondition object.
	 */
	public Object clone()
	{
		RuleCondition c = new RuleCondition(values.length);
		c.values = (double[])this.values.clone();
		return c;
	}

	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////
	
}
