package ucs;

import dataprocessing.Example;

/**
 * A population member for use in UCS.
 * @author Gavin Brown
 *
 */
public class Indiv implements Cloneable
{
	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////

	/** A reference to the UCSconfig object defining the workings of the current UCS run. */
	protected UCSconfig params;
	
	/** The RuleCondition object defining the matching input space for this rule. */
	public RuleCondition condition;
	
	/** The action suggested by this rule */
	public double action;

	/** The total number of possible actions. */
	public static int numOutputs;
	
	///////////////////////////////////////
	
	/** The current accuracy of this rule. */
	public double accuracy;
	
	/** The number of examples correctly classified by this rule */
	public int numCorrect;

	/** The number of examples matched by this rule, also known as <i>experience</i> in the LCS literature. */
	public int numMatches; //aka experience;

	/** A timestamp saying how long is has been since this individual was given the chance to be mutated/crossed over. */
	protected int lastTimeThisWasInTheGA;
	
	/** The <i>numerosity</i> of this individual.  This is always set as 1, since macroclassifiers are not currently supported.
	 * Future versions of the UCS packages may use it. */
	protected int numerosity=1; //for macroclassifiers
	
	/** Average size of the correct sets that this rule participates in. */
	public int correctSetSize;
	
	///////////////////////////////////////////////////
	///////////////////////////////////////////////////
	///////////////////////////////////////////////////
	///////////////////////////////////////////////////

        public Indiv() {
	    throw new IllegalStateException("Cannot use the default constructor for Indiv");
	}

	/** Constructor for this class.
	 * @param length The length of the conditions to be used in this run.
	 * @param conf A <i>UCSconfig</i> object specifying the workings of this UCS run.
	 */
	public Indiv( int length, UCSconfig conf )
	{
		this.params = (UCSconfig)conf;
		this.condition = new RuleCondition(length);
		
		// 0=0, 1=1, #=2
		for (int i=0; i<length; i++)
		{
			// random number from the valid range including #
			condition.set( i, condition.randomValue(i) );						
		}
				
		//random action from the valid range
		numOutputs = Example.targetAlphabet.size();
		action = params.generator.nextInt(numOutputs);

		//this is how UCS initialises new rules. Orriols' thesis p.44
		//Actually he maintains fitness as a parameter while it's a function here but perhaps that
		//doesn't make any difference.
		// FIX -- some of these values are also set in clone() which is redundant
		accuracy=1;
		numMatches = 1;
		numCorrect = 1;
		numerosity = 1;
	}
		
	///////////////////////////////////////////////////
	///////////////////////////////////////////////////
	///////////////////////////////////////////////////
	///////////////////////////////////////////////////

	
	/** Returns the fitness of this individual, determined by the current UCSconfig object.
	 * @return	The fitness, exactly as returned by the call: params.fitfunc.evaluate(this).
	 */
    //	public double fitness()
    //	{
    //		return params.fitfunc.evaluate(this);
    //	}
	
	///////////////////////////////////////////////////
	///////////////////////////////////////////////////
	///////////////////////////////////////////////////
	///////////////////////////////////////////////////

	/** Checks whether the provided input is matched by this individual.
	 * @param input An input to check.
	 * @return True if this individual matches the input, otherwise false. If the system
	 * is not in TESTMODE, an update is made to this individual's numMatches count.
	 */
	public boolean matches( double[] input )
	{	
   	        if (numMatches < 1) {
		    throw new IllegalStateException("numMatches < 1");
		}
	    
		if( !condition.matches(input) ) return false;
		
		if( !UCS.TESTMODE ) 
		{
				numMatches++;
				accuracy = numCorrect/(double)numMatches;
		}
		
		return true;
	}

	///////////////////////////////////////////////////
	///////////////////////////////////////////////////
	///////////////////////////////////////////////////
	///////////////////////////////////////////////////
	
	/** Checks whether the action suggested by this individual matches the provided action.
	 * @param correctAction An action.
	 * @return True if this individual's action matches the correctAction, otherwise false. If the system
	 * is not in TESTMODE, an update is made to this individual's numCorrect count.
	 * 
	 */
	protected boolean test( double correctAction )
	{
		if(this.action != correctAction) return false;

		
		if( !UCS.TESTMODE ) 
		{
				numCorrect++;
				accuracy = numCorrect/(double)numMatches;
		}
		
		return true;
	}
	
	///////////////////////////////////////////////////
	///////////////////////////////////////////////////
	///////////////////////////////////////////////////
	///////////////////////////////////////////////////

	/** Updates the running average of the size of
	 * the correct set in which this rule participates 
	 * @param s The current correct set size, used to update the previous ones in a running average.
	*/
	protected void updateCorrectSetSize(int s)
	{
		//maintains a running average of the size of
		//the correct set in which this rule participates
		//
		correctSetSize = (int)((correctSetSize*(numCorrect-1) + s) / (double)numCorrect);
	}
	
	///////////////////////////////////////////////////
	///////////////////////////////////////////////////
	///////////////////////////////////////////////////
	
	/** Performs one-point crossover and single point mutation between this individual and the provided individual.
	 *  @return An array of two children produced by the
	 * operations.  
	 * ``The parameters of the offspring are
	 * initialized as follows. The experience and the numerosity
	 * are set to 1. The accuracy and the fitness are also set to
	 * 1 (these parameters will go quickly to their real values as
	 * they participate in successive match sets). Finally, cs is
	 * set to the size of the current correct set.'' Thesis of Albert Orriols p.44
	 * Note it does not say how to initialise rules generated by covering.
	 */
	protected Indiv[] crossAndMutate( Indiv parent )
	{
		double otherGenome[] = parent.condition.values;
		
		Indiv [] child = new Indiv[2];
		//System.out.println("child[0]=" + child[0]);
		child[0] = new Indiv(otherGenome.length, this.params);
		child[1] = new Indiv(otherGenome.length, this.params);
		
		
		//if (UCS.iter>10000) crossoverProb=0;
		if(params.generator.nextDouble() < params.crossoverProb)
		{
			//CROSSOVER THEIR GENOMES
			//
			double crossPoint = params.generator.nextInt(otherGenome.length);
			for (int i=0; i<condition.values.length; i++)
			{
				int tmp=0;
				if(i < crossPoint)
				{
					//I'm the daddy
					child[0].condition.values[i] = this.condition.values[i];
					child[1].condition.values[i] = otherGenome[i];
				}
				else
				{
					//I'm the mommy
					child[0].condition.values[i] = otherGenome[i];
					child[1].condition.values[i] = this.condition.values[i];
				}
			}
			
			//AVERAGE THEIR PARAMETERS
			//
			for (int c=0; c<2; c++)
			{
				child[c].correctSetSize = (int)((this.correctSetSize + parent.correctSetSize)/2.0);
				child[c].accuracy = (this.accuracy + parent.accuracy)/2.0;
				child[c].action = this.action;
			}
		}
		else
		{
			child[0] = (Indiv)parent.clone();
			child[1] = (Indiv)this.clone();
		}
		

		//MUTATION
		//
		//flip the genome bits with certain probabilities.
		//
		for (int c=0; c<2; c++)
		{
			RuleCondition condition = child[c].condition;
			
			for (int i=0; i<condition.values.length; i++)
			{
				if(params.generator.nextDouble() < params.mutationProb)
				{
					double newAllele = condition.randomValue(i);
					
					while (newAllele == child[c].condition.values[i])
						newAllele = condition.randomValue(i);
						
					condition.values[i] = newAllele;
				}
			}

			//flip the action with same probability.
			//
			if(params.generator.nextDouble() < params.mutationProb)
			{
				int newAction = params.generator.nextInt(numOutputs);
				while (newAction == child[c].action) newAction = params.generator.nextInt(numOutputs);
				child[c].action = newAction;
			}
		}

		if (child[0].numMatches == 0) throw new IllegalStateException("numMatches == 0 in crossAndMutate");
		if (child[1].numMatches == 0) throw new IllegalStateException("numMatches == 0 in crossAndMutate");
		return child;
	}
	
	
	///////////////////////////////////////////////////
	///////////////////////////////////////////////////
	///////////////////////////////////////////////////
	///////////////////////////////////////////////////

	/** Checks whether this individual logically subsumes the supplied rule.
	 * @param An individual to check against.
	 * @return True if this rule logically subsumes the supplied rule.
	 */
	protected boolean subsumes( Indiv child )
	{
		Indiv parent = this;
		
		//NOTE: b1=this, b2=other
		//
		boolean parentMoreGeneral=false;
	    for (int i=0; i<parent.condition.length; i++)
	    {
	        // if bits are the same, test next pair
	        if (parent.condition.values[i] == child.condition.values[i]) continue;

	        // if b1 has a don't care and b2 doesn't then b1 is more general. test next pair
	        if (parent.condition.values[i]==RuleCondition.HASH)
	        {
	        	parentMoreGeneral = true;
	            continue;
	        }

	        // if b1 and b2 have different bits and b1 does not have a don't care, it does not subsume b2
		    parentMoreGeneral = false;
		    break;
	    }    
	    
		
	    if(	parent.numMatches > this.params.ThetaSub
	    &&	parent.accuracy > this.params.ThetaSubAccuracyMinimum
	    &&	parentMoreGeneral)
	    {
			return true;
		}
		
		return false;
	}
	
	///////////////////////////////////////////////////
	///////////////////////////////////////////////////
	///////////////////////////////////////////////////
	///////////////////////////////////////////////////
	
	/** Clones this Indiv object.
	 * @return An exact copy of this object.
	 * FIX - what is the point of this method? It's confusing. We should never want an
	 * exact copy of an individual, and this doesn't actually produce one. Is it part of a hack
	 * to simulate numerosity?
	 */
	public Object clone()
	{
		Indiv copy = new Indiv(this.condition.values.length, this.params);
	
		copy.action = this.action;
		copy.accuracy = 1;//this.accuracy;
		copy.numCorrect = 1;//this.numCorrect;
		copy.numMatches = 1;//this.numMatches;
		copy.numOutputs = this.numOutputs;
		copy.numerosity = this.numerosity;
		copy.correctSetSize = this.correctSetSize;
		copy.lastTimeThisWasInTheGA = this.lastTimeThisWasInTheGA;
		copy.condition = (RuleCondition)this.condition.clone();
		
		return copy;
	}
	
	///////////////////////////////////////////////////
	///////////////////////////////////////////////////
	///////////////////////////////////////////////////
	///////////////////////////////////////////////////

	/**
	 * Returns a string representation of this object.
	 * @return A String representation of this Individual, using '#' as the "don't care" symbol.
	 */
	public String toString()
	{
		String s="";
		for (int i=0; i<condition.values.length; i++)
			if(condition.values[i]==RuleCondition.HASH) s+= " #";
			//else s+=(int)condition.values[i];
			else s+=" "+Example.inputAlphabet[i].get((int)condition.values[i]);

		//s+=":"+(int)action;
		s+=" : "+Example.targetAlphabet.get((int)action);
		
		//NOTE: THE RETURN VALUE FROM THIS MUST BE ONLY THE RULE AS THIS METHOD
		//IS USED TO CALCULATE THE NUMBER OF MACROCLASSIFIERS
		//
		return s;
	}
	
	///////////////////////////////////////////////////
	///////////////////////////////////////////////////
	///////////////////////////////////////////////////
	///////////////////////////////////////////////////
	
}
