function bagFrames = RunLeedsBaggageDetector (allPeopleFile, varargin)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% RunLeedsBaggageDetector
% Syntax:
%   bags = RunLeedsBaggageDetector (peopleFile, 'propName1', 'propValue1', ...)
%
% Descrioption:
%   returns bounding boxes on all the detected bags
%
% Main Properties:
%   backgroundColour         The colour chosen for the background in the binary images (Default: 0 - black)
%   homography               A file with the ground-plane homography in the following format
%   imgtype                  The format of the foreground images (Default: jpg)
%   minHeight                Minimum accepted height of an individual to detect bags (Default 80)
%   verticalLines            A file with a few vertical lines (perpendicular to the ground plane) to calculate the position of the camera in the image plane.
%   writeIntermediateResults Outputs the 
%
% Other Properties
%   maxNoOfFrames Maximum number of frames (when split = true) (Default 50)
%   minNoOfFrames Minimum number of frames in the trajectory (Default 10)
%   personHeight  Height of an individual in pixels. Larger areas are rescaled. Set to 240.
%   personWidth   Width of the individual. Set to 160     
%   PRange        ??
%   split         Boolean (true/false). When true, any trajectory longer than maxNoOfFrames is split into shorter trajectories. (Default false)
%   topValue      The percentage of non-noisy frames used to create the temporal template. (Default 80 - i.e. 80\% of the frames are used)
%   
% Reference:
%   Damen, Hogg (2007). Detecting Carried Objects from Short Video Sequences.
%   European Conference on Computer Vision (ECCV).
%
% Other information
%   Project: Leeds Baggage Detector
%   Author : Dima J. Damen 
%   Creation Date : Mon August 27th 2007
%
% COPYRIGHT
%{
%This source file is the copyright property of the University of Leeds
%('The University').
%
%Permission to use, copy, modify, and distribute this source file for
%educational, research, and not-for-profit purposes, without fee and
%without a signed licensing agreement, is hereby granted, provided that
%the above copyright notice, this paragraph and the following three
%paragraphs appear in all copies, modifications, and distributions.
%
%In no event shall The University be liable to any party for direct,
%indirect, special, incidental or consequential damages, including lost
%profits, arising out of the use of this software and its documentation.
%
%The software is provided without warranty. The University has no
%obligation to provide maintenance, support, updates, enhancements, or
%modifications.
%
%This software was written by Dima Damen, Vision Group,
%School of Computing, University of Leeds, UK. The code and use
%thereof should be attributed to the author where appropriate
%(including demonstrations which rely on it's use).
%}
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

   % loads the parameter values
   params = loadParameters ();
   params.parse (varargin {:});

   % loads the data file
   file = load (allPeopleFile);
   minNoOfFrames = 10;
   
   % calculate the homography information
   path (path, 'Hom');
   homFile = load (params.Results.homography);
   H = homography2d (homFile);
   [cameraX, cameraY] = FindVanishingPointInWorld2 (params.Results.verticalLines, H);

   % split the trajectories if needed
   if (params.Results.split == true)
       file = convertToSplits_f (file);
       dlmwrite (strcat ('3_', allPeopleFile), file, 'precision', '%g');
   end;
   
   [m,n] = size (file);
   bagFrames = zeros (0,6);
   
   % loops through all people.
   for i=1:m
      tic;
      setNo = file(i,1);
      objNo = file(i,2);
      startFrame = file(i,3);
      endFrame = file(i,4);
      % shows the object Number for tracking purposes
      objNo
      
      if (endFrame - startFrame < minNoOfFrames)
          continue;
      end;      
      
      warning off all;
      beginningFileName = strcat ('Set', num2str(setNo), '\', num2str (objNo));
      boundingBoxesFile = load (strcat ('Set', num2str(setNo), '\', num2str(objNo), '_BB.dat'));
      rescaleFactor = decideRescaleFactor (boundingBoxesFile, params);
      [normalizedTemplate, newMedianFile, centredImages, foundFrames, footLoc] = getNormalizedPersonTemplate_f (beginningFileName, setNo, startFrame, endFrame, boundingBoxesFile, rescaleFactor, params);
      [s, normalizedTemplate] = recentralizeTemplate2 (beginningFileName, normalizedTemplate, params);
      [direction, trans, diff] = analyzePersonWithTemplate_f (beginningFileName, normalizedTemplate, newMedianFile, params, H, cameraX, cameraY, rescaleFactor, footLoc);
      [pBagProb, pNoBagProb] = multiplyPersonDiffWithPriors (beginningFileName, diff, direction, trans, 'Set1To8Prior', params);
      bags = calculatePersonBags (beginningFileName, pBagProb, pNoBagProb, normalizedTemplate, params) ;
      outFile = ReflectBBOntoImages2 (newMedianFile, bags, s, rescaleFactor, centredImages, foundFrames, params);
      outFile(:,8) = objNo;
      bagFrames = [bagFrames; outFile];
   end
end